#!/usr/bin/perl

# Guy Bruneau, gbruneau@ipss.ca
#
# Date: 15 Aug 2017
# Version: 1.0
#
# Updated: 11 March 2022
# Version: 1.1
# 
# All files must me in the same directory before executing the Perl script:
# perl ip2asn.pl  ip2asn.tsv
#
# Usage: ./ip2asn.pl
#
# This script is used to parse the ip2asn.csv from iptoasn.com to be later
# load into RSA NetWitness to map AS Number.

# In order to parse ip2asn.csv correctly, download it from the following link:
#
# Download IPv4 file from: https://iptoasn.com
# wget https://iptoasn.com/data/ip2asn-v4.tsv.gz
# gunzip ip2asn-v4.tsv.gz
# Unzip the file and then run this command to convert the file into comma delimited:
# cat ip2asn-v4.tsv | sed 's/\t/,/g' > ip2asn.tsv
#
# After your ran the script against the CSV file, use NwConsole to compile the feed
# or upload it via the GUI to the decoders. To compile it via NwConsole:
# NwConsole
# feed create ip2asn.xml
# Upload the feed ip2asn.feed to all decoders

# You will need to add the followin keys to both Broker/Concentrator and restart 
# nwbroker and nwconcentrator services
#
# <key description="Source ASN" format="UInt32" level="IndexValues" name="asn.src" valueMax="500000"/>
# <key description="Destination ASN" format="UInt32" level="IndexValues" name="asn.dst" valueMax="500000"/>
#
# Add the following to each decoders and restart decoder service
# <key description="Source ASN" name="asn.src" format="UInt32" level="IndexNone"/>
# <key description="Destination ASN" name="asn.dst" format="UInt32" level="IndexNone"/>

# Required Perl packages

use IO::Handle;
use Getopt::Long;
use IO::Socket;

# Global variables

my $SrcIP;
my $DstIP;
my $ASN;

# This subroutine processes the GeoIPASNum2.csv CSV formated file into a format
# a format that will be converted into a NetWitness feed file.

open(OUTFILE, ">ip2asn.csv") or die "Error (output): $!";

print OUTFILE $message;

close OUTFILE;

# Open the GeoIP ASN file and 2 output files to create 2 map files
# One file for source address ranges and one for destination address ranges

open(INFILE, "ip2asn.tsv") or die "Error (input): $!";
open(OUTFILE, ">>ip2asn.csv") or die "Error (output): $!";


while (<INFILE>) {

   @parsedLine = split;

    IPFixed ();
}

# This subroutine parses the GeoIPASNum2.csv IP addresses (Src and Dst) into a format
# a format that can be imported in RSA NetWitness decoder as a feed.
 
sub IPFixed () {

   $field1 = $parsedLine[0];
   ($SrcIP,$DstIP,$ASN) = split(/\,/, $field1);

   # Remove " from the ASN number
   $ASN =~ s/\"AS|AS//;

   my @Net_List = splice @parsedLine, 1, 35;
   $NetOwner = join(" ", @Net_List);
   $NetOwner =~ s/\"//g;

   # Add comment Not Listed if a  record does not have a ASN owner listed
   # to prevent parsing errors

   if ($NetOwner eq "") {
      $NetOwner = "Not Listed";
   }

   # Returns the Src and Dst addresses

   $SRCIP = inet_ntoa(inet_aton($SrcIP));
   $DSTIP = inet_ntoa(inet_aton($DstIP));
  
# Format the output before sending it to file

   PrintFormat ();

}

sub PrintFormat () {

# This subroutine prints the results of the data processing into the correct format.

# $SRCIP -> Start IP network range
# $DSTIP -> Destination IP network range
# $ASN -> Network AS Number

   $message = ("$SRCIP,$DSTIP" . ",$ASN" . "\n");


# Output data into master file

print OUTFILE $message;

}

# Close the files

close OUTFILE;
close INFILE;

